const express = require("express");
const router = express.Router();
const {
  createTherapyPlan,
  getTherapistPlans,
  getClientPlans,
  getAllPlans,
  getTherapyPlan,
  updateTherapyPlan,
  sendTherapyPlan,
  respondToTherapyPlan,
  archiveTherapyPlan,
  getTherapyPlanStats,
  getTherapistClients
} = require("../controller/therapyPlanController");
const { verifyToken, isAdmin, isDoctor, isUser } = require("../middleware/authMiddleware");

console.log("🔧 Therapy Plan routes being registered...");

// Therapist routes (require doctor role)
router.post("/create", verifyToken, isDoctor, createTherapyPlan); // Create new therapy plan
router.get("/therapist/plans", verifyToken, isDoctor, getTherapistPlans); // Get therapist's plans
router.get("/therapist/clients", verifyToken, isDoctor, getTherapistClients); // Get therapist's clients
router.put("/:planId", verifyToken, isDoctor, updateTherapyPlan); // Update therapy plan
router.post("/:planId/send", verifyToken, isDoctor, sendTherapyPlan); // Send plan to client

// Client routes (require user role)
router.get("/client/plans", verifyToken, isUser, getClientPlans); // Get client's plans
router.post("/:planId/respond", verifyToken, isUser, respondToTherapyPlan); // Client respond to plan

// Admin routes (require admin role)
router.get("/admin/all", verifyToken, isAdmin, getAllPlans); // Get all plans (admin only)

// Debug route to check user authentication
router.get("/debug/user", verifyToken, (req, res) => {
  res.json({
    success: true,
    user: {
      id: req.user.id,
      role: req.user.role,
      email: req.user.email
    }
  });
});

// Simple test endpoint without authentication
router.get("/test", (req, res) => {
  res.json({
    success: true,
    message: "Therapy plans routes are working",
    timestamp: new Date().toISOString()
  });
});

// Shared routes (any authenticated user)
router.get("/stats", verifyToken, getTherapyPlanStats); // Get statistics
router.get("/:planId", verifyToken, getTherapyPlan); // Get specific plan
router.post("/:planId/archive", verifyToken, archiveTherapyPlan); // Archive plan

module.exports = router;
